<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\TrainingModel;

class TrainingController extends BaseController
{
    protected $trainingModel;

    public function __construct()
    {
        $this->trainingModel = new TrainingModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $filter = [
            'penyelenggara' => $this->request->getGet('penyelenggara') ?? '',
            'tanggal_mulai' => $this->request->getGet('tanggal_mulai') ?? '',
            'tanggal_selesai' => $this->request->getGet('tanggal_selesai') ?? ''
        ];
        $sort = $this->request->getGet('sort') ?? 'urutan';
        $order = $this->request->getGet('order') ?? 'ASC';
        $page = (int)($this->request->getGet('page') ?? 1);

        $builder = $this->trainingModel->searchAndFilter($search, $filter, $sort, $order);
        $total = $builder->countAllResults(false);
        
        $training = $builder->get(10, ($page - 1) * 10)->getResultArray();

        $pager = \Config\Services::pager();
        $pager->store('training', $page, 10, $total);

        $data = [
            'title' => 'Kelola Pelatihan',
            'training' => $training,
            'pager' => $pager,
            'search' => $search,
            'filter' => $filter,
            'sort' => $sort,
            'order' => $order
        ];

        return view('admin/training/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Pelatihan'
        ];
        return view('admin/training/create', $data);
    }

    public function store()
    {
        $rules = [
            'nama_pelatihan' => 'required|min_length[3]|max_length[200]',
            'sertifikat' => 'permit_empty|uploaded[sertifikat]|max_size[sertifikat,5120]|ext_in[sertifikat,jpg,jpeg,png,pdf]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileSertifikat = $this->request->getFile('sertifikat');
        $sertifikatName = null;

        if ($fileSertifikat && $fileSertifikat->isValid() && !$fileSertifikat->hasMoved()) {
            $sertifikatName = $fileSertifikat->getRandomName();
            $fileSertifikat->move(WRITEPATH . 'uploads/training/', $sertifikatName);
        }

        $data = [
            'nama_pelatihan' => $this->request->getPost('nama_pelatihan'),
            'penyelenggara' => $this->request->getPost('penyelenggara'),
            'tanggal_mulai' => $this->request->getPost('tanggal_mulai'),
            'tanggal_selesai' => $this->request->getPost('tanggal_selesai'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'sertifikat' => $sertifikatName,
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($this->trainingModel->insert($data)) {
            return redirect()->to('admin/training')->with('success', 'Pelatihan berhasil ditambahkan!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan pelatihan!');
        }
    }

    public function edit($id)
    {
        $training = $this->trainingModel->find($id);
        
        if (!$training) {
            return redirect()->to('admin/training')->with('error', 'Pelatihan tidak ditemukan!');
        }

        $data = [
            'title' => 'Edit Pelatihan',
            'training' => $training
        ];
        return view('admin/training/edit', $data);
    }

    public function update($id)
    {
        $training = $this->trainingModel->find($id);
        
        if (!$training) {
            return redirect()->to('admin/training')->with('error', 'Pelatihan tidak ditemukan!');
        }

        $rules = [
            'nama_pelatihan' => 'required|min_length[3]|max_length[200]',
            'sertifikat' => 'permit_empty|uploaded[sertifikat]|max_size[sertifikat,5120]|ext_in[sertifikat,jpg,jpeg,png,pdf]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $fileSertifikat = $this->request->getFile('sertifikat');
        
        $data = [
            'nama_pelatihan' => $this->request->getPost('nama_pelatihan'),
            'penyelenggara' => $this->request->getPost('penyelenggara'),
            'tanggal_mulai' => $this->request->getPost('tanggal_mulai'),
            'tanggal_selesai' => $this->request->getPost('tanggal_selesai'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($fileSertifikat && $fileSertifikat->isValid() && !$fileSertifikat->hasMoved()) {
            if ($training['sertifikat'] && file_exists(WRITEPATH . 'uploads/training/' . $training['sertifikat'])) {
                unlink(WRITEPATH . 'uploads/training/' . $training['sertifikat']);
            }
            $sertifikatName = $fileSertifikat->getRandomName();
            $fileSertifikat->move(WRITEPATH . 'uploads/training/', $sertifikatName);
            $data['sertifikat'] = $sertifikatName;
        }

        if ($this->trainingModel->update($id, $data)) {
            return redirect()->to('admin/training')->with('success', 'Pelatihan berhasil diupdate!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate pelatihan!');
        }
    }

    public function delete($id)
    {
        $training = $this->trainingModel->find($id);
        
        if (!$training) {
            return redirect()->to('admin/training')->with('error', 'Pelatihan tidak ditemukan!');
        }

        if ($training['sertifikat'] && file_exists(WRITEPATH . 'uploads/training/' . $training['sertifikat'])) {
            unlink(WRITEPATH . 'uploads/training/' . $training['sertifikat']);
        }

        if ($this->trainingModel->delete($id)) {
            return redirect()->to('admin/training')->with('success', 'Pelatihan berhasil dihapus!');
        } else {
            return redirect()->to('admin/training')->with('error', 'Gagal menghapus pelatihan!');
        }
    }
}


